#!/usr/bin/env python3
"""
Validate fly.toml configuration before deployment.
Catches common issues that cause deployment failures.

Usage: python validate_fly_toml.py [path/to/fly.toml]
"""

import sys
import tomllib
from pathlib import Path


def load_toml(path: Path) -> dict:
    """Load and parse fly.toml."""
    try:
        with open(path, "rb") as f:
            return tomllib.load(f)
    except FileNotFoundError:
        return {"_error": f"File not found: {path}"}
    except tomllib.TOMLDecodeError as e:
        return {"_error": f"Invalid TOML syntax: {e}"}


def validate(config: dict) -> list[str]:
    """Validate fly.toml configuration. Returns list of issues."""
    issues = []
    
    if "_error" in config:
        return [config["_error"]]
    
    # Required: app name
    if "app" not in config:
        issues.append("Missing 'app' - app name is required")
    
    # Required: primary_region
    if "primary_region" not in config:
        issues.append("Missing 'primary_region' - deployment region is required")
    
    # Check build configuration
    build = config.get("build", {})
    has_dockerfile = "dockerfile" in build
    has_image = "image" in build
    has_builder = "builder" in build
    
    if not (has_dockerfile or has_image or has_builder):
        # Check if Dockerfile exists in current directory
        if not Path("Dockerfile").exists():
            issues.append(
                "No build configuration found. Add [build] section with "
                "'dockerfile', 'image', or 'builder', or create a Dockerfile"
            )
    
    # Check http_service or services
    http_service = config.get("http_service", {})
    services = config.get("services", [])
    
    if not http_service and not services:
        issues.append(
            "No service configuration found. Add [http_service] for HTTP apps "
            "or [[services]] for other protocols"
        )
    
    # Validate http_service
    if http_service:
        internal_port = http_service.get("internal_port")
        if internal_port is None:
            issues.append("[http_service] missing 'internal_port'")
        elif not isinstance(internal_port, int) or internal_port < 1 or internal_port > 65535:
            issues.append(f"[http_service] internal_port must be 1-65535, got: {internal_port}")
        
        # Check auto_stop/auto_start consistency
        auto_stop = http_service.get("auto_stop_machines")
        auto_start = http_service.get("auto_start_machines")
        
        if auto_stop and auto_stop != "off" and auto_start is False:
            issues.append(
                "Warning: auto_stop_machines is enabled but auto_start_machines is false. "
                "Machines may stop and not restart on traffic."
            )
        
        # Check health checks
        checks = http_service.get("checks", [])
        for i, check in enumerate(checks):
            if "path" not in check:
                issues.append(f"[http_service.checks[{i}]] missing 'path'")
            grace = check.get("grace_period", "0s")
            if grace == "0s":
                issues.append(
                    f"[http_service.checks[{i}]] grace_period is 0s - "
                    "app may fail health checks before it starts"
                )
    
    # Validate services sections
    for i, service in enumerate(services):
        if "internal_port" not in service:
            issues.append(f"[[services]][{i}] missing 'internal_port'")
        if "protocol" not in service:
            issues.append(f"[[services]][{i}] missing 'protocol' (tcp or udp)")
        
        ports = service.get("ports", [])
        if not ports:
            issues.append(f"[[services]][{i}] has no [[services.ports]] entries")
    
    # Check mounts for volume existence
    mounts = config.get("mounts")
    if mounts:
        source = mounts.get("source")
        if source:
            issues.append(
                f"Note: Using volume '{source}'. Ensure volume exists: "
                f"fly volumes create {source} --region <region>"
            )
    
    # Check deploy section
    deploy = config.get("deploy", {})
    release_cmd = deploy.get("release_command")
    if release_cmd:
        timeout = deploy.get("release_command_timeout", "5m")
        issues.append(
            f"Note: release_command configured ('{release_cmd}'). "
            f"Timeout is {timeout}. Increase if migrations are slow."
        )
    
    # Check env for common mistakes
    env = config.get("env", {})
    for key in env:
        if key.startswith("FLY_"):
            issues.append(
                f"[env] key '{key}' starts with FLY_ which is reserved for Fly.io runtime"
            )
        if "SECRET" in key.upper() or "PASSWORD" in key.upper() or "KEY" in key.upper():
            issues.append(
                f"Warning: [env] contains '{key}' which looks sensitive. "
                "Use 'fly secrets set' instead of [env] for secrets."
            )
    
    return issues


def main():
    path = Path(sys.argv[1]) if len(sys.argv) > 1 else Path("fly.toml")
    
    print(f"Validating: {path}")
    print("-" * 40)
    
    config = load_toml(path)
    issues = validate(config)
    
    if not issues:
        print("✓ No issues found")
        return 0
    
    errors = [i for i in issues if not i.startswith(("Note:", "Warning:"))]
    warnings = [i for i in issues if i.startswith("Warning:")]
    notes = [i for i in issues if i.startswith("Note:")]
    
    for issue in errors:
        print(f"✗ {issue}")
    for issue in warnings:
        print(f"⚠ {issue}")
    for issue in notes:
        print(f"ℹ {issue}")
    
    print("-" * 40)
    print(f"Errors: {len(errors)}, Warnings: {len(warnings)}, Notes: {len(notes)}")
    
    return 1 if errors else 0


if __name__ == "__main__":
    sys.exit(main())
